// =========================
// content.js - Live Monitoring without Force Zoom, Window Resize 738×766 after 2Captcha submission, then 380×305 after final submit
// =========================

// 1) Debug: log when content.js is injected
console.log('🚀 [content.js] Script loaded with live monitoring.');

// 2) Selectors for Money1.html
const FIND_INSTRUCTION_SELECTOR   = '.comment-row.decrease-comment-xs';
const CAPTCHA_CONTAINER_SELECTOR  = '.oldsolver';
const SUBMIT_BUTTON_SELECTOR      = 'button[data-v-4acb3785].btn.btn-submit';

// 3) Global variables for tracking
let processedImages    = new Set();
let isProcessing       = false;
let observer           = null;
let currentCaptchaId   = null;
let submitClickTime    = null;
let isPaused           = false;
let lastSubmissionTime = 0;
let pauseTimeout       = null;
let checkTimeout       = null;
let lastCheckTime      = 0;

// 4) Track original window size so we can restore it later
let originalWindowWidth  = null;
let originalWindowHeight = null;

// 5) Enhanced message sending with retry logic
async function sendMessageToBackground(message, retries = 3) {
  for (let attempt = 1; attempt <= retries; attempt++) {
    console.log(`🌐 [content.js] Sending message (attempt ${attempt}/${retries}):`, message.type);
    try {
      const response = await new Promise((resolve, reject) => {
        const timeout = setTimeout(() => reject(new Error('Message timeout')), 30000);
        chrome.runtime.sendMessage(message, (response) => {
          clearTimeout(timeout);
          if (chrome.runtime.lastError)         reject(new Error(chrome.runtime.lastError.message));
          else if (!response)                   reject(new Error('No response from background'));
          else                                   resolve(response);
        });
      });
      console.log(`✅ [content.js] Got response (attempt ${attempt}):`, response);
      return response;
    } catch (error) {
      console.error(`❗ [content.js] Message failed (attempt ${attempt}):`, error);
      if (attempt === retries) throw error;
      await new Promise(res => setTimeout(res, 1000 * attempt));
    }
  }
}

// 6) Utilities to parse and extract Base64
function parseTargetObject(text) {
  const match = text.match(/Find object\s*"(.*?)"/i);
  return match ? match[1] : null;
}
function extractBase64(styleValue) {
  const m = styleValue.match(/url\(\s*['"]?data:image\/[^;]+;base64,([^'")]+)['"]?\)/i);
  return m ? m[1] : null;
}

// 7) Create hash for duplicate detection
function createImageHash(base64) {
  let hash = 0;
  for (let i = 0; i < Math.min(base64.length, 1000); i++) {
    const char = base64.charCodeAt(i);
    hash = ((hash << 5) - hash) + char;
    hash &= hash;
  }
  return hash.toString();
}

// 8) Create custom instruction for 2Captcha
function createCustomInstruction(targetObject) {
  if (!targetObject) return "Click on all relevant objects in the image";
  const instruction = `Click on all images with ${targetObject}`;
  console.log(`📝 [content.js] Created instruction: "${instruction}"`);
  return instruction;
}

// 9) Get actual image dimensions from base64
function getImageDimensions(base64) {
  return new Promise((resolve, reject) => {
    const img = new Image();
    img.onload = () => resolve({ width: img.width, height: img.height });
    img.onerror = reject;
    img.src = `data:image/png;base64,${base64}`;
  });
}

// 10) Request window resize via background
function requestWindowResize(width, height) {
  chrome.runtime.sendMessage(
    { type: 'resizeWindow', width: width, height: height },
    (resp) => {
      if (!resp || !resp.success) {
        console.warn('⚠️ [content.js] resizeWindow message failed:', resp && resp.error);
      } else {
        console.log(`✅ [content.js] Requested window resize to ${width}×${height}`);
      }
    }
  );
}

// 11) Calculate coordinate scaling factors (fallback)
function calculateScaling(containerEl, imageWidth, imageHeight) {
  const rect = containerEl.getBoundingClientRect();
  const computedStyle = window.getComputedStyle(containerEl);
  const bgSize = computedStyle.backgroundSize;
  let scaleX = 1, scaleY = 1;
  console.log('📏 [content.js] Calculating scaling:', {
    containerSize: { width: rect.width, height: rect.height },
    imageSize: { width: imageWidth, height: imageHeight },
    backgroundSize: bgSize
  });
  if (bgSize === 'contain') {
    scaleX = scaleY = Math.min(rect.width / imageWidth, rect.height / imageHeight);
    console.log('📏 [content.js] Background-size: contain - using uniform scaling');
  } else {
    scaleX = rect.width / imageWidth;
    scaleY = rect.height / imageHeight;
    console.log('📏 [content.js] Background-size: default - using stretch scaling');
  }
  console.log('📏 [content.js] Final scaling factors:', {
    scaleX: scaleX.toFixed(4),
    scaleY: scaleY.toFixed(4)
  });
  return { scaleX, scaleY };
}

// 12) Auto-click coordinates and submit
async function autoClickCoordinatesAndSubmit(coordString, containerEl, base64Image) {
  console.log('🎯 [content.js] Auto-clicking coordinates (no force zoom)...');
  if (!coordString || !containerEl) {
    console.error('❗ [content.js] Invalid parameters for auto-clicking');
    return;
  }
  try {
    // 12a) Center the CAPTCHA container
    containerEl.scrollIntoView({ block: 'center', inline: 'center' });

    // 12b) Parse coordString into pairs
    let pairs = [];
    if (typeof coordString === 'string') {
      pairs = coordString.split('|').map(p => {
        const coords = p.split(',').map(Number);
        if (coords.length !== 2 || isNaN(coords[0]) || isNaN(coords[1])) {
          throw new Error(`Invalid coordinate pair: ${p}`);
        }
        return coords;
      });
    } else if (Array.isArray(coordString)) {
      pairs = coordString.map(coord => {
        if (coord && typeof coord === 'object' && coord.x !== undefined && coord.y !== undefined) {
          return [coord.x, coord.y];
        } else if (Array.isArray(coord) && coord.length >= 2) {
          return [coord[0], coord[1]];
        } else if (typeof coord === 'string') {
          const nums = coord.split(',').map(Number);
          if (nums.length >= 2) return [nums[0], nums[1]];
        }
        throw new Error(`Invalid coordinate format: ${JSON.stringify(coord)}`);
      });
    } else if (coordString && typeof coordString === 'object') {
      if (coordString.x !== undefined && coordString.y !== undefined) {
        pairs = [[coordString.x, coordString.y]];
      } else {
        throw new Error(`Invalid coordinate object: ${JSON.stringify(coordString)}`);
      }
    }

    if (pairs.length === 0) {
      console.warn('⚠️ [content.js] No valid coordinates found');
      return;
    }

    // 12c) Compute scaling (image→ container size)
    let scaleX = 1, scaleY = 1;
    if (base64Image) {
      try {
        const imageDims = await getImageDimensions(base64Image);
        const scaling = calculateScaling(containerEl, imageDims.width, imageDims.height);
        scaleX = scaling.scaleX;
        scaleY = scaling.scaleY;
      } catch {
        console.warn('⚠️ [content.js] Could not determine scaling, using 1:1');
      }
    }

    // 12d) Show highlight markers for 10 seconds
    const markers = [];
    pairs.forEach(([x, y], index) => {
      const scaledX = x * scaleX;
      const scaledY = y * scaleY;
      const marker = document.createElement('div');
      marker.style.cssText = `
        position: absolute;
        left: ${scaledX - 10}px;
        top: ${scaledY - 10}px;
        width: 20px;
        height: 20px;
        border: 2px solid #00ff00;
        border-radius: 50%;
        background-color: rgba(0, 255, 0, 0.7);
        pointer-events: none;
        z-index: 999999;
      `;
      containerEl.appendChild(marker);
      markers.push(marker);
      console.log(`🎯 [content.js] Will click coordinate ${index + 1}: (${scaledX.toFixed(1)}, ${scaledY.toFixed(1)})`);
    });
    setTimeout(() => {
      markers.forEach(m => m.remove());
      console.log('📍 [content.js] Removed highlight markers after 10 seconds');
    }, 10000);

    // 12e) Dispatch clicks with random 0.5–1s delay between each
    for (let i = 0; i < pairs.length; i++) {
      const [x, y] = pairs[i];
      const scaledX = x * scaleX;
      const scaledY = y * scaleY;

      // Re-center container before each click
      containerEl.scrollIntoView({ block: 'center', inline: 'center' });
      await new Promise(res => setTimeout(res, 100)); // allow scroll to settle

      const rect = containerEl.getBoundingClientRect();
      const absoluteX = rect.left + scaledX;
      const absoluteY = rect.top + scaledY;

      console.log(`🎯 [content.js] Clicking coordinate ${i + 1}:`);
      console.log(`  Original: (${x}, ${y})`);
      console.log(`  Scaled: (${scaledX.toFixed(1)}, ${scaledY.toFixed(1)})`);
      console.log(`  Container rect:`, rect);
      console.log(`  Absolute position: (${absoluteX.toFixed(1)}, ${absoluteY.toFixed(1)})`);

      // Method 1: MouseEvent on container
      const clickEvent = new MouseEvent('click', {
        bubbles: true,
        cancelable: true,
        view: window,
        clientX: absoluteX,
        clientY: absoluteY,
        screenX: absoluteX,
        screenY: absoluteY,
        button: 0,
        buttons: 1
      });
      containerEl.dispatchEvent(clickEvent);

      // Method 2: Also click nested element at point
      const elementAtPoint = document.elementFromPoint(absoluteX, absoluteY);
      if (elementAtPoint && elementAtPoint !== containerEl) {
        elementAtPoint.dispatchEvent(clickEvent);
        console.log(`  ✓ Also clicked nested element:`, elementAtPoint.tagName);
      }

      // Method 3: mousedown & mouseup
      const mouseDownEvent = new MouseEvent('mousedown', {
        bubbles: true,
        cancelable: true,
        view: window,
        clientX: absoluteX,
        clientY: absoluteY,
        button: 0,
        buttons: 1
      });
      const mouseUpEvent = new MouseEvent('mouseup', {
        bubbles: true,
        cancelable: true,
        view: window,
        clientX: absoluteX,
        clientY: absoluteY,
        button: 0,
        buttons: 1
      });
      containerEl.dispatchEvent(mouseDownEvent);
      containerEl.dispatchEvent(mouseUpEvent);

      console.log(`✅ [content.js] Dispatched all click events for coordinate ${i + 1}`);

      // Random 0.5–1s pause before next click
      if (i < pairs.length - 1) {
        const randomClickDelay = 500 + Math.random() * 500;
        await new Promise(res => setTimeout(res, randomClickDelay));
      }
    }

    // 12f) Reposition submit button to bottom center so it's fully visible
    const submitButton = document.querySelector(SUBMIT_BUTTON_SELECTOR);
    if (submitButton) {
      submitButton.style.position   = 'fixed';
      submitButton.style.bottom     = '20px';
      submitButton.style.left       = '50%';
      submitButton.style.transform  = 'translateX(-50%)';
      submitButton.style.zIndex     = '1000000';
      await new Promise(res => setTimeout(res, 100)); // allow reposition to render

      // Random 0.5–1s delay before clicking “Submit”
      const randomSubmitDelay = 500 + Math.random() * 500;
      await new Promise(res => setTimeout(res, randomSubmitDelay));

      submitButton.click();
      submitClickTime = Date.now();
      console.log('✅ [content.js] Clicked submit button, monitoring for login redirect...');

      // 12g) After 5 seconds, resize window to 380×305 via background
      setTimeout(() => {
        requestWindowResize(380, 305);
      }, 5000);

      monitorForLoginRedirect();
    } else {
      console.warn('⚠️ [content.js] Submit button not found');
    }

  } catch (error) {
    console.error('❗ [content.js] Error in auto-click:', error);
  }
}

// 13) Monitor for login redirect after submit
function monitorForLoginRedirect() {
  if (!currentCaptchaId || !submitClickTime) {
    console.log('🔍 [content.js] No captcha ID or submit time to monitor');
    return;
  }
  const checkInterval = 2000;
  const maxDuration   = 40000;
  console.log(`🕐 [content.js] Monitoring for login redirect (40s max)...`);

  const checkForLogin = () => {
    const elapsed = Date.now() - submitClickTime;
    if (
      window.location.pathname.includes('/login') ||
      window.location.href.includes('/login') ||
      document.title.toLowerCase().includes('login')
    ) {
      console.log('❌ [content.js] Login redirect detected – reporting incorrect');
      reportIncorrectCaptcha(currentCaptchaId);
      return;
    }
    if (elapsed < maxDuration) {
      setTimeout(checkForLogin, checkInterval);
    } else {
      console.log('✅ [content.js] No login redirect detected – CAPTCHA likely correct');
      currentCaptchaId  = null;
      submitClickTime  = null;
    }
  };

  setTimeout(checkForLogin, checkInterval);
}

// 14) Report incorrect captcha to 2Captcha
async function reportIncorrectCaptcha(captchaId) {
  console.log(`📣 [content.js] Reporting incorrect captcha ID: ${captchaId}`);
  try {
    const response = await sendMessageToBackground({ type: 'reportIncorrect', captchaId });
    if (response.success) {
      console.log('✅ [content.js] Successfully reported incorrect captcha');
    } else {
      console.error('❗ [content.js] Failed to report incorrect captcha:', response.error);
    }
  } catch (error) {
    console.error('❗ [content.js] Error reporting incorrect captcha:', error);
  } finally {
    currentCaptchaId  = null;
    submitClickTime  = null;
  }
}

// 15) Main CAPTCHA solving logic with smart pausing
async function solveAndAutoSubmitCaptcha() {
  if (isPaused) {
    console.log('⏸️ [content.js] In pause mode, skipping...');
    return;
  }
  if (isProcessing) {
    console.log('⏳ [content.js] Already processing, skipping...');
    return;
  }
  console.log('🕵️ [content.js] solveAndAutoSubmitCaptcha() started.');
  isProcessing = true;

  try {
    // 15a) Find the “Find object …” instruction
    const instrEl = document.querySelector(FIND_INSTRUCTION_SELECTOR);
    if (!instrEl) {
      console.log('🔍 [content.js] No instruction found, exiting.');
      return;
    }
    const instrText = instrEl.textContent.trim();
    const target    = parseTargetObject(instrText);
    if (!target) {
      console.warn(`⚠️ [content.js] Cannot parse target from "${instrText}".`);
      return;
    }
    console.log(`🖼️ [content.js] Instruction: Find object "${target}"`);

    // 15b) Locate CAPTCHA container and extract Base64
    const captchaDiv = document.querySelector(CAPTCHA_CONTAINER_SELECTOR);
    if (!captchaDiv) {
      console.error('❗ [content.js] Cannot find CAPTCHA container.');
      return;
    }
    const style = window.getComputedStyle(captchaDiv);
    const bg    = style.getPropertyValue('background-image');
    const base64= extractBase64(bg);
    if (!base64) {
      console.error('❗ [content.js] Failed to extract base64 from background-image.');
      return;
    }

    // 15c) Avoid re-processing the same image
    const imageHash = createImageHash(base64);
    if (processedImages.has(imageHash)) {
      console.log('🔄 [content.js] Image already processed, skipping...');
      return;
    }
    processedImages.add(imageHash);
    console.log(`✅ [content.js] Processing new image (hash: ${imageHash})`);

    // 15d) Store original window size before any resizing
    originalWindowWidth  = window.outerWidth;
    originalWindowHeight = window.outerHeight;
    console.log(`🖥️ [content.js] Original window size: ${originalWindowWidth}×${originalWindowHeight}`);

    // 15e) Send Base64 + custom instruction to background (2Captcha)
    const customInstruction = createCustomInstruction(target);
    console.log('🌐 [content.js] Sending to 2Captcha...');
    const submitResult = await sendMessageToBackground({
      type:        'submitCaptcha',
      body:        base64,
      instruction: customInstruction
    });
    if (!submitResult.success) {
      console.error('❗ [content.js] 2Captcha submission failed:', submitResult.error);
      return;
    }
    const captchaId = submitResult.captchaId;
    currentCaptchaId = captchaId;
    console.log(`✅ [content.js] Received captchaId: ${captchaId}`);

    // 15f) Immediately request window resize to 738×766
    requestWindowResize(900, 900);

    // 15g) Poll for coordinates
    console.log('⏳ [content.js] Polling for coords...');
    const pollResult = await sendMessageToBackground({
      type:      'pollCaptcha',
      captchaId: captchaId
    });
    if (!pollResult.success) {
      console.error('❗ [content.js] Polling failed:', pollResult.error);
      return;
    }
    const coords = pollResult.coords;
    console.log(`✅ [content.js] coords = "${coords}"`);

    // 15h) Auto-click coordinates and submit
    await autoClickCoordinatesAndSubmit(coords, captchaDiv, base64);
    console.log('🎉 [content.js] Auto-submission complete!');

    // 15i) Start 10-minute pause
    startPauseMode();

  } catch (error) {
    console.error('❗ [content.js] Fatal error:', error);
  } finally {
    isProcessing = false;
  }
}

// 15j) Start pause mode after submission
function startPauseMode() {
  const pauseDuration = 10 * 60 * 1000; // 10 minutes
  lastSubmissionTime = Date.now();
  isPaused = true;
  console.log('⏸️ [content.js] Starting 10-minute pause mode...');
  console.log(`⏸️ [content.js] Will resume at: ${new Date(Date.now() + pauseDuration).toLocaleTimeString()}`);
  if (pauseTimeout) clearTimeout(pauseTimeout);
  pauseTimeout = setTimeout(() => {
    isPaused = false;
    console.log('▶️ [content.js] Pause mode ended – resuming monitoring');
    console.log('🔄 [content.js] Checking for new CAPTCHAs…');
    setTimeout(solveAndAutoSubmitCaptcha, 1000);
  }, pauseDuration);
}

// 16) Live monitoring with throttling
function startLiveMonitoring() {
  console.log('👁️ [content.js] Starting live monitoring…');
  setTimeout(solveAndAutoSubmitCaptcha, 2000);

  observer = new MutationObserver((mutations) => {
    if (isPaused || isProcessing) return;
    const now = Date.now();
    const timeSinceLastCheck = now - lastCheckTime;
    if (timeSinceLastCheck < 3000) {
      console.log(`⏱️ [content.js] Throttled – last check ${(timeSinceLastCheck/1000).toFixed(1)}s ago`);
      return;
    }
    if (checkTimeout) {
      console.log(`⏱️ [content.js] Check already scheduled, skipping…`);
      return;
    }
    let shouldCheck = false;
    for (const mutation of mutations) {
      if (mutation.type === 'childList' && mutation.addedNodes.length > 0) shouldCheck = true;
      else if (mutation.type === 'attributes' && (mutation.attributeName === 'style' || mutation.attributeName === 'class')) shouldCheck = true;
    }
    if (shouldCheck) {
      console.log('🔄 [content.js] DOM change detected, scheduling check in 1s…');
      checkTimeout = setTimeout(() => {
        checkTimeout = null;
        lastCheckTime = Date.now();
        console.log('🕵️ [content.js] Executing scheduled check…');
        solveAndAutoSubmitCaptcha();
      }, 1000);
    }
  });

  // Backup periodic check every 5s
  setInterval(() => {
    if (!isPaused && !isProcessing && !checkTimeout) {
      const now = Date.now();
      if (now - lastCheckTime >= 5000) {
        console.log('🔄 [content.js] Periodic check (backup)…');
        lastCheckTime = now;
        solveAndAutoSubmitCaptcha();
      }
    }
  }, 5000);

  observer.observe(document.body, {
    childList: true,
    subtree: true,
    attributes: true,
    attributeFilter: ['style', 'class']
  });

  console.log('✅ [content.js] Live monitoring active');
}

// 17) Cleanup function
function stopLiveMonitoring() {
  if (observer) {
    observer.disconnect();
    observer = null;
    console.log('🛑 [content.js] Live monitoring stopped');
  }
  if (pauseTimeout) {
    clearTimeout(pauseTimeout);
    pauseTimeout = null;
  }
  if (checkTimeout) {
    clearTimeout(checkTimeout);
    checkTimeout = null;
  }
}

// 18) Manual testing functions
window.testCaptchaCoordinates = function(coordString) {
  console.log('🧪 [content.js] Manual coordinate test:', coordString);
  const container = document.querySelector(CAPTCHA_CONTAINER_SELECTOR);
  if (!container) {
    console.error('❗ [content.js] CAPTCHA container not found for manual test');
    return;
  }
  autoClickCoordinatesAndSubmit(coordString, container, null);
};

window.solveCaptchaNow = function() {
  console.log('🎯 [content.js] Manual captcha solve triggered');
  solveAndAutoSubmitCaptcha();
};

window.pauseCaptcha = function() {
  startPauseMode();
  console.log('⏸️ [content.js] Manual pause triggered');
};

window.resumeCaptcha = function() {
  isPaused = false;
  if (pauseTimeout) {
    clearTimeout(pauseTimeout);
    pauseTimeout = null;
  }
  console.log('▶️ [content.js] Manual resume triggered');
};

window.testClick = function(x, y) {
  console.log(`🎯 [content.js] Testing manual click at (${x}, ${y})`);
  const container = document.querySelector(CAPTCHA_CONTAINER_SELECTOR);
  if (!container) {
    console.error('❗ [content.js] CAPTCHA container not found');
    return;
  }
  // No force zoom
  container.scrollIntoView({ block: 'center', inline: 'center' });
  setTimeout(() => {
    const rect = container.getBoundingClientRect();
    const absoluteX = rect.left + x;
    const absoluteY = rect.top + y;
    console.log(`Container rect:`, rect);
    console.log(`Absolute click position: (${absoluteX.toFixed(1)}, ${absoluteY.toFixed(1)})`);
    const events = [
      new MouseEvent('mousedown', { bubbles: true, clientX: absoluteX, clientY: absoluteY }),
      new MouseEvent('click',     { bubbles: true, clientX: absoluteX, clientY: absoluteY }),
      new MouseEvent('mouseup',   { bubbles: true, clientX: absoluteX, clientY: absoluteY })
    ];
    events.forEach(event => {
      container.dispatchEvent(event);
      console.log(`Dispatched ${event.type} event`);
    });
    const elementAtPoint = document.elementFromPoint(absoluteX, absoluteY);
    if (elementAtPoint) {
      console.log('Element at point:', elementAtPoint);
      elementAtPoint.click();
    }
  }, 300);
};

// 19) Handle extension context invalidation
chrome.runtime.onConnect.addListener(() => {
  console.log('🔗 [content.js] Runtime connection established');
});

// 20) Initialize when DOM is ready
function init() {
  console.log('🔄 [content.js] Initializing live monitoring extension...');
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', startLiveMonitoring);
  } else {
    startLiveMonitoring();
  }
  window.addEventListener('beforeunload', stopLiveMonitoring);
}

// Start the extension
init();
