// =========================
// background.js - ULTRA FAST Version with 30s timeout, auto-reporting, and window resize support
// =========================

// 1) Debug: Log when the service worker starts
console.log('🚀 [background.js] Service worker started.');

// 2) Your 2Captcha API key
const TWO_CAPTCHA_API_KEY = 'ceac2804fca24900976b6b082cb0462b';

// 3) FAST polling interval - 2 seconds instead of 5
const POLL_INTERVAL_MS = 2000; // 2 seconds for faster response

// 4) Maximum wait time for CAPTCHA solution - 30 seconds
const MAX_WAIT_TIME_MS = 30000; // 30 seconds maximum

// 5) Keep service worker alive
chrome.runtime.onStartup.addListener(() => {
  console.log('🔄 [background.js] Extension startup detected');
});
chrome.runtime.onInstalled.addListener(() => {
  console.log('🔧 [background.js] Extension installed/updated');
});

// 6) Message listener
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('📨 [background.js] Received message:', message);

  // Handle window resize requests separately
  if (message.type === 'resizeWindow') {
    chrome.windows.getCurrent((win) => {
      if (!win || !win.id) {
        sendResponse({ success: false, error: 'Could not get window ID' });
        return;
      }
      chrome.windows.update(win.id, {
        width: message.width,
        height: message.height
      }, () => {
        if (chrome.runtime.lastError) {
          console.error('❗ [background.js] resizeWindow failed:', chrome.runtime.lastError.message);
          sendResponse({ success: false, error: chrome.runtime.lastError.message });
        } else {
          console.log(`✅ [background.js] Window resized to ${message.width}×${message.height}`);
          sendResponse({ success: true });
        }
      });
    });
    return true; // Keep message channel open for async response
  }

  // Handle the other message types
  (async () => {
    try {
      if (message.type === 'submitCaptcha') {
        const base64Image = message.body;
        const customInstruction = message.instruction || null;
        console.log('🔄 [background.js] Calling submitTo2Captcha()...');
        console.log('📝 [background.js] Custom instruction:', customInstruction);

        const captchaId = await submitTo2Captcha(base64Image, customInstruction);
        console.log(`✅ [background.js] Received captchaId: ${captchaId}`);

        sendResponse({ success: true, captchaId });
      }
      else if (message.type === 'pollCaptcha') {
        const captchaId = message.captchaId;
        console.log(`🔄 [background.js] Calling pollForCoordinates() for ID: ${captchaId}...`);

        try {
          const coords = await pollForCoordinates(captchaId);
          console.log(`✅ [background.js] Received coords: ${coords}`);
          sendResponse({ success: true, coords });
        } catch (error) {
          console.error(`❗ [background.js] Polling failed for ID ${captchaId}:`, error);

          // Auto-report as incorrect if polling times out
          if (error.message.includes('timeout') || error.message.includes('30 seconds')) {
            console.log(`🚨 [background.js] Auto-reporting incorrect due to timeout: ${captchaId}`);
            try {
              await reportIncorrectCaptcha(captchaId);
              console.log(`✅ [background.js] Auto-reported incorrect captcha: ${captchaId}`);
            } catch (reportError) {
              console.error(`❗ [background.js] Failed to auto-report incorrect:`, reportError);
            }
          }

          sendResponse({ success: false, error: error.message, captchaId });
        }
      }
      else if (message.type === 'reportIncorrect') {
        const captchaId = message.captchaId;
        console.log(`📣 [background.js] Reporting incorrect captcha ID: ${captchaId}...`);

        const result = await reportIncorrectCaptcha(captchaId);
        console.log(`✅ [background.js] Report result:`, result);

        sendResponse({ success: true, result });
      }
      else if (message.type === 'ping') {
        console.log('🏓 [background.js] Ping received');
        sendResponse({ success: true, message: 'pong' });
      }
      else {
        console.warn('⚠️ [background.js] Unknown message type:', message.type);
        sendResponse({ success: false, error: 'Unknown message type' });
      }
    } catch (err) {
      console.error('❗ [background.js] Error caught:', err);
      sendResponse({ success: false, error: err.message });
    }
  })();

  return true; // Keep message channel open for async response
});

// 7) Submit to 2Captcha
async function submitTo2Captcha(base64Image, customInstruction = null) {
  console.log('🌐 [background.js] submitTo2Captcha: building FormData...');

  if (!base64Image) {
    throw new Error('Base64 image is required');
  }
  if (!TWO_CAPTCHA_API_KEY || TWO_CAPTCHA_API_KEY === 'your_api_key_here') {
    throw new Error('2Captcha API key not configured');
  }

  try {
    const form = new FormData();
    form.append('method', 'base64');
    form.append('key', TWO_CAPTCHA_API_KEY);
    form.append('body', base64Image);
    form.append('coordinatescaptcha', '1');
    form.append('json', '1');
    if (customInstruction) {
      form.append('textinstructions', customInstruction);
      console.log('📝 [background.js] Added textinstructions:', customInstruction);
    }

    console.log('🌐 [background.js] Making request to 2captcha.com/in.php...');
    const response = await fetch('https://2captcha.com/in.php', {
      method: 'POST',
      body: form,
      headers: { 'User-Agent': 'Chrome Extension' }
    });
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    const data = await response.json();
    console.log('🌐 [background.js] 2Captcha /in.php response:', data);
    if (data.status === 1) {
      return data.request; // the captchaId
    } else {
      throw new Error(`2Captcha submission failed: ${data.request || 'Unknown error'}`);
    }
  } catch (error) {
    console.error('❗ [background.js] submitTo2Captcha error:', error);
    throw error;
  }
}

// 8) ULTRA FAST Poll for coordinates with 30-second timeout
async function pollForCoordinates(captchaId) {
  console.log(`🔃 [background.js] pollForCoordinates started for ID: ${captchaId} (max 30 seconds)`);

  if (!captchaId) {
    throw new Error('Captcha ID is required');
  }
  const baseUrl = 'https://2captcha.com/res.php';
  const maxAttempts = Math.ceil(MAX_WAIT_TIME_MS / POLL_INTERVAL_MS);
  const startTime = Date.now();
  let attempts = 0;

  console.log(`⏰ [background.js] Will wait maximum ${MAX_WAIT_TIME_MS / 1000} seconds (${maxAttempts} attempts)`);

  while (attempts < maxAttempts) {
    attempts++;
    const elapsed = Date.now() - startTime;
    if (elapsed >= MAX_WAIT_TIME_MS) {
      const timeoutError = `CAPTCHA polling timeout after ${MAX_WAIT_TIME_MS / 1000} seconds (${attempts} attempts)`;
      console.error(`⏰ [background.js] ${timeoutError}`);
      throw new Error(timeoutError);
    }
    console.log(`🔃 [background.js] Polling attempt ${attempts}/${maxAttempts} (${(elapsed/1000).toFixed(1)}s elapsed)`);
    try {
      if (attempts > 1) {
        await new Promise(res => setTimeout(res, POLL_INTERVAL_MS));
      }
      const currentElapsed = Date.now() - startTime;
      if (currentElapsed >= MAX_WAIT_TIME_MS) {
        const timeoutError = `CAPTCHA polling timeout after ${MAX_WAIT_TIME_MS / 1000} seconds during wait`;
        console.error(`⏰ [background.js] ${timeoutError}`);
        throw new Error(timeoutError);
      }
      const url = `${baseUrl}?key=${TWO_CAPTCHA_API_KEY}&action=get&id=${captchaId}&json=1`;
      console.log(`🌐 [background.js] Fetching: ${url}`);
      const resp = await fetch(url, { headers: { 'User-Agent': 'Chrome Extension' } });
      if (!resp.ok) {
        throw new Error(`HTTP error! status: ${resp.status}`);
      }
      const result = await resp.json();
      console.log('🌐 [background.js] /res.php response:', result);
      if (result.status === 1) {
        const finalElapsed = Date.now() - startTime;
        console.log(`🎉 [background.js] CAPTCHA solved after ${attempts} attempts in ${(finalElapsed/1000).toFixed(1)}s`);
        let coordinates = result.request;
        if (Array.isArray(coordinates)) {
          console.log('📝 [background.js] Coordinates returned as array:', coordinates);
          if (coordinates.length > 0 && typeof coordinates[0] === 'object') {
            coordinates = coordinates.map(coord => {
              if (coord.x !== undefined && coord.y !== undefined) {
                return `${coord.x},${coord.y}`;
              } else if (Array.isArray(coord) && coord.length >= 2) {
                return `${coord[0]},${coord[1]}`;
              } else {
                return String(coord);
              }
            }).join('|');
          } else {
            coordinates = coordinates.join('|');
          }
        } else if (typeof coordinates === 'object' && coordinates !== null) {
          console.log('📝 [background.js] Coordinates returned as object:', coordinates);
          if (coordinates.x !== undefined && coordinates.y !== undefined) {
            coordinates = `${coordinates.x},${coordinates.y}`;
          } else {
            coordinates = JSON.stringify(coordinates);
          }
        } else if (typeof coordinates !== 'string') {
          coordinates = String(coordinates);
        }
        console.log('✅ [background.js] Final coordinates string:', coordinates);
        return coordinates;
      } else if (result.status === 0 && (result.request === 'CAPCHA_NOT_READY' || result.request === 'CAPTCHA_NOT_READY')) {
        const remainingTime = MAX_WAIT_TIME_MS - (Date.now() - startTime);
        console.log(`⏳ [background.js] CAPTCHA not ready. ${(remainingTime/1000).toFixed(1)}s remaining...`);
        continue;
      } else {
        throw new Error(`2Captcha error: ${result.request || 'Unknown error'}`);
      }
    } catch (error) {
      const currentElapsed = Date.now() - startTime;
      console.error(`❗ [background.js] Polling attempt ${attempts} failed (${(currentElapsed/1000).toFixed(1)}s elapsed):`, error);
      if (attempts >= maxAttempts || currentElapsed >= MAX_WAIT_TIME_MS) {
        const finalError = `Polling failed after ${MAX_WAIT_TIME_MS / 1000} seconds (${attempts} attempts): ${error.message}`;
        console.error(`⏰ [background.js] ${finalError}`);
        throw new Error(finalError);
      }
    }
  }

  const finalError = `Polling timeout after ${MAX_WAIT_TIME_MS / 1000} seconds (${maxAttempts} attempts)`;
  console.error(`⏰ [background.js] ${finalError}`);
  throw new Error(finalError);
}

// 9) Report incorrect captcha
async function reportIncorrectCaptcha(captchaId) {
  console.log(`📣 [background.js] reportIncorrectCaptcha for ID: ${captchaId}`);
  if (!captchaId) {
    throw new Error('Captcha ID is required for reporting');
  }
  if (!TWO_CAPTCHA_API_KEY || TWO_CAPTCHA_API_KEY === 'your_api_key_here') {
    throw new Error('2Captcha API key not configured');
  }
  try {
    // Try the reportIncorrect API first
    console.log('🌐 [background.js] Trying reportIncorrect API...');
    const requestBody = {
      clientKey: TWO_CAPTCHA_API_KEY,
      taskId:    parseInt(captchaId)
    };
    console.log('🌐 [background.js] Sending reportIncorrect request:', requestBody);
    const response = await fetch('https://api.2captcha.com/reportIncorrect', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'User-Agent':    'Chrome Extension'
      },
      body: JSON.stringify(requestBody)
    });
    if (response.ok) {
      const data = await response.json();
      console.log('🌐 [background.js] reportIncorrect API response:', data);
      if (data.errorId === 0 && data.status === 'success') {
        console.log('✅ [background.js] Successfully reported incorrect captcha via API');
        return data;
      }
    }
    // Fallback to the classic reportbad method
    console.log('🌐 [background.js] API failed, trying classic reportbad method...');
    const fallbackUrl = `https://2captcha.com/res.php?key=${TWO_CAPTCHA_API_KEY}&action=reportbad&id=${captchaId}&json=1`;
    console.log('🌐 [background.js] Fallback URL:', fallbackUrl);
    const fallbackResponse = await fetch(fallbackUrl, {
      method:  'GET',
      headers: { 'User-Agent': 'Chrome Extension' }
    });
    if (!fallbackResponse.ok) {
      throw new Error(`HTTP error! status: ${fallbackResponse.status}`);
    }
    const fallbackData = await fallbackResponse.json();
    console.log('🌐 [background.js] reportbad response:', fallbackData);
    if (fallbackData.status === 1 && fallbackData.request === 'OK_REPORT_RECORDED') {
      console.log('✅ [background.js] Successfully reported incorrect captcha via reportbad');
      return fallbackData;
    } else {
      throw new Error(`reportbad failed: ${fallbackData.request || 'Unknown error'}`);
    }
  } catch (error) {
    console.error('❗ [background.js] reportIncorrectCaptcha error:', error);
    console.warn('⚠️ [background.js] Failed to report incorrect captcha, but continuing...');
    return { error: error.message };
  }
}
